%% =========================================================
% Supplementary Material
%% =========================================================

clearvars
close all
clc

years = 2020:2023;
nYears = numel(years);
nBoot  = 1000;

%% =========================================================
% Load yearly and total sedimentation data
%% =========================================================

Sed_OPEN  = cell(1,nYears);
Sed_MOSE  = cell(1,nYears);
Sed_ATHOS = cell(1,nYears);

for i = 1:nYears
    y = years(i);
    
    load(['Sed_' num2str(y) '_OPEN.mat']);
    Sed_OPEN{i} = eval(['Sed_' num2str(y) '_OPEN']);
    
    load(['Sed_' num2str(y) '_MOSE.mat']);
    Sed_MOSE{i} = eval(['Sed_' num2str(y) '_MOSE']);
    
    load(['Sed_' num2str(y) '_ATHOS.mat']);
    Sed_ATHOS{i} = eval(['Sed_' num2str(y) '_ATHOS']);
end

load Sed_OPEN_TOTAL.mat
load Sed_MOSE_TOTAL.mat
load Sed_ATHOS_TOTAL.mat

%% =========================================================
% Convert sedimentation to sedimentation rate
rho_s = 960; % Kg/m³ from Puppin et al., 2025
% - TOTAL: averaged over 4 years
% - YEARLY: single-year rates
%% =========================================================

SR_OPEN_TOTAL  = (Sed_OPEN_TOTAL  ./ rho_s) * 1000 / 4;
SR_MOSE_TOTAL  = (Sed_MOSE_TOTAL  ./ rho_s) * 1000 / 4;
SR_ATHOS_TOTAL = (Sed_ATHOS_TOTAL ./ rho_s) * 1000 / 4;

SR_OPEN  = cell(1,nYears);
SR_MOSE  = cell(1,nYears);
SR_ATHOS = cell(1,nYears);

for i = 1:nYears
    SR_OPEN{i}  = (Sed_OPEN{i}  ./ rho_s) * 1000;
    SR_MOSE{i}  = (Sed_MOSE{i}  ./ rho_s) * 1000;
    SR_ATHOS{i} = (Sed_ATHOS{i} ./ rho_s) * 1000;
end

%% =========================================================
% FIGURE 1: Total Sedimentation (OPEN vs MOSE)
%% =========================================================

figure(1); clf

% ---------- KDE (left panel) ----------
subplot(1,2,1); hold on

xi = linspace( ...
    min([SR_OPEN_TOTAL, SR_MOSE_TOTAL]) - 1, ...
    max([SR_OPEN_TOTAL, SR_MOSE_TOTAL]) + 1, 200);

for k = 1:2
    if k == 1
        data = SR_OPEN_TOTAL; col = 'b';
    else
        data = SR_MOSE_TOTAL; col = 'r';
    end
    
    boot = zeros(nBoot,length(xi));
    for j = 1:nBoot
        boot(j,:) = ksdensity(datasample(data,numel(data)),xi);
    end
    
    CI = prctile(boot,[2.5 97.5]);
    f  = mean(boot);
    
    fill([xi fliplr(xi)], [CI(1,:) fliplr(CI(2,:))], ...
        col,'FaceAlpha',0.2,'EdgeColor','none');
    plot(xi,f,col,'LineWidth',2)
end

xlabel('Sedimentation Rate [mm yr^{-1} m^{-2}]')
ylabel('Probability Density (-)')
xlim([0 20]); ylim([0 0.25])
title('KDE: OPEN vs MOSE')
legend('95% CI OPEN','OPEN','95% CI MOSE','MOSE')

% ---------- ECDF (right panel) ----------
subplot(1,2,2); hold on

[fo,xo,flo,fuo] = ecdf(Sed_OPEN_TOTAL,'Bounds','on');
[fm,xm,flm,fum] = ecdf(Sed_MOSE_TOTAL,'Bounds','on');

plot(xo,1-fo,'b','LineWidth',1.5)
plot(xm,1-fm,'r','LineWidth',1.5)
plot(xo,1-flo,'--b'); plot(xo,1-fuo,'--b')
plot(xm,1-flm,'--r'); plot(xm,1-fum,'--r')

set(gca,'YScale','log')
ylim([1e-5 1]); xlim([0 350])
xlabel('Total Sedimentation [kg m^{-2}]')
ylabel('Exceedance Probability (-)')
title('ECDF: OPEN vs MOSE')

%% =========================================================
% FIGURE 2: Total Sedimentation (MOSE vs ATHOS)
%% =========================================================

figure(2); clf

subplot(1,2,1); hold on

xi = linspace( ...
    min([SR_MOSE_TOTAL, SR_ATHOS_TOTAL]) - 1, ...
    max([SR_MOSE_TOTAL, SR_ATHOS_TOTAL]) + 1, 200);

datasets = {SR_MOSE_TOTAL, SR_ATHOS_TOTAL};
colors   = {'r','g'};

for k = 1:2
    data = datasets{k};
    col  = colors{k};
    
    boot = zeros(nBoot,length(xi));
    for j = 1:nBoot
        boot(j,:) = ksdensity(datasample(data,numel(data)),xi);
    end
    
    CI = prctile(boot,[2.5 97.5]);
    f  = mean(boot);
    
    fill([xi fliplr(xi)], [CI(1,:) fliplr(CI(2,:))], ...
        col,'FaceAlpha',0.2,'EdgeColor','none');
    plot(xi,f,col,'LineWidth',2)
end

xlabel('Sedimentation Rate [mm yr^{-1} m^{-2}]')
ylabel('Probability Density (-)')
xlim([0 20]); ylim([0 0.25])
title('KDE: MOSE vs ATHOS')
legend('95% CI MOSE','MOSE','95% CI ATHOS','ATHOS')
subplot(1,2,2); hold on

[fm,xm,flm,fum] = ecdf(Sed_MOSE_TOTAL,'Bounds','on');
[fa,xa,fla,fua] = ecdf(Sed_ATHOS_TOTAL,'Bounds','on');

plot(xm,1-fm,'r','LineWidth',1.5)
plot(xa,1-fa,'g','LineWidth',1.5)
plot(xm,1-flm,'--r'); plot(xm,1-fum,'--r')
plot(xa,1-fla,'--g'); plot(xa,1-fua,'--g')

set(gca,'YScale','log')
ylim([1e-5 1]); xlim([0 350])
xlabel('Total Sedimentation [kg m^{-2}]')
ylabel('Exceedance Probability (-)')
title('ECDF: MOSE vs ATHOS')

%% =========================================================
% FIGURE 3: Yearly Sedimentation (2020–2023)
% Row 1: ECDF (Sed)
% Row 2: KDE bootstrap (SR)
%% =========================================================

figure(3); clf

for i = 1:nYears
    
    % ---------- ECDF ----------
    subplot(2,4,i); hold on
    
    [fo,xo,flo,fuo] = ecdf(Sed_OPEN{i}, 'Bounds','on');
    [fm,xm,flm,fum] = ecdf(Sed_MOSE{i}, 'Bounds','on');
    [fa,xa,fla,fua] = ecdf(Sed_ATHOS{i},'Bounds','on');
    
    plot(xo,1-fo,'b','LineWidth',1.2)
    plot(xm,1-fm,'r','LineWidth',1.2)
    plot(xa,1-fa,'g','LineWidth',1.2)
    
    plot(xo,1-flo,'--b'); plot(xo,1-fuo,'--b')
    plot(xm,1-flm,'--r'); plot(xm,1-fum,'--r')
    plot(xa,1-fla,'--g'); plot(xa,1-fua,'--g')
    
    set(gca,'YScale','log')
    ylim([1e-5 1]); xlim([0 150])
    title(['Year ' num2str(years(i))])
    xlabel('Yearly Sedimentation [kg m^{-2}]')
    ylabel('Exceedance Probability (-)')
    % ---------- KDE ----------
    subplot(2,4,i+4); hold on
    
    xi = linspace( ...
        min([SR_OPEN{i}, SR_MOSE{i}, SR_ATHOS{i}]) - 1, ...
        max([SR_OPEN{i}, SR_MOSE{i}, SR_ATHOS{i}]) + 1, 200);
    
    datasets = {SR_OPEN{i}, SR_MOSE{i}, SR_ATHOS{i}};
    colors   = {'b','r','g'};
    
    for k = 1:3
        data = datasets{k};
        col  = colors{k};
        
        boot = zeros(nBoot,length(xi));
        for j = 1:nBoot
            boot(j,:) = ksdensity(datasample(data,numel(data)),xi);
        end
        
        CI = prctile(boot,[2.5 97.5]);
        f  = mean(boot);
        
        fill([xi fliplr(xi)], [CI(1,:) fliplr(CI(2,:))], ...
            col,'FaceAlpha',0.2,'EdgeColor','none');
        plot(xi,f,col,'LineWidth',1.2)
    end
    xlabel('Sedimentation Rate [mm m^{-2}]')
    ylabel('Probability Density (-)')
    xlim([0 20]); ylim([0 0.25])
end
legend('95% CI OPEN','OPEN','95% CI MOSE','MOSE','95% CI ATHOS','ATHOS')